<?php
// classes/WdoiHelper.inc.php

use APP\facades\Repo;               // OJS 3.4 repo facade

class WdoiHelper
{
  /** Pick localized string from array/object or scalar */
  public static function pickLocalized($value, array $preferredLocales) {
    if (!is_array($value)) {
      if (is_string($value)) return $value;
      if (is_object($value) && method_exists($value, '__toString')) return (string)$value;
      return '';
    }
    foreach ($preferredLocales as $loc) {
      if (!empty($value[$loc]) && is_string($value[$loc])) return $value[$loc];
    }
    foreach ($value as $v) {
      if (is_string($v) && $v !== '') return $v;
    }
    return '';
  }

  /** Preferred locales order: UI → Context primary → Site primary → en_US */
  public static function preferredLocales($request) {
    $list = [];
    try { $ui = \AppLocale::getLocale(); if ($ui) $list[] = $ui; } catch (\Throwable $e) {}
    try {
      $ctx = $request ? $request->getContext() : null;
      $cp  = $ctx ? $ctx->getPrimaryLocale() : null;
      if ($cp) $list[] = $cp;
    } catch (\Throwable $e) {}
    try { $sp = \AppLocale::getPrimaryLocale(); if ($sp) $list[] = $sp; } catch (\Throwable $e) {}
    $list[] = 'en_US';
    return array_values(array_unique(array_filter($list)));
  }

  /** Normalize ORCID to https://orcid.org/XXXX-XXXX-XXXX-XXXX */
  private static function normalizeOrcid(?string $orcid): ?string {
    $o = trim((string)($orcid ?? ''));
    if ($o === '') return null;
    // Strip URI/prefixes
    $o = preg_replace('~^https?://(www\.)?orcid\.org/~i', '', $o);
    $o = preg_replace('~[^0-9Xx-]~', '', $o);
    return 'https://orcid.org/' . $o;
  }

  /** Map authors to [{given, family, affiliation?, orcid?}] */
  public static function authorsForPublication($publicationId, $contextId, array $preferredLocales) {
    $out = [];
    try {
      // Collector returns authors; we sort by sequence ascending to preserve order
      $collector = Repo::author()->getCollector();
      $collector->filterByPublicationIds([(int)$publicationId]);
      $collector->orderBy('seq', 'ASC');
      $authors = $collector->getMany();

      foreach ($authors as $a) {
        $given  = self::pickLocalized($a->getGivenName(null), $preferredLocales);
        $family = self::pickLocalized($a->getFamilyName(null), $preferredLocales);
        $aff    = self::pickLocalized($a->getAffiliation(null), $preferredLocales);
        $orcid  = self::normalizeOrcid($a->getData('orcid'));

        $row = ['given' => $given, 'family' => $family];
        if ($aff)   $row['affiliation'] = $aff;
        if ($orcid) $row['orcid'] = $orcid;
        $out[] = $row;
      }
    } catch (\Throwable $e) {
      error_log('[wDOI] authors Repo fallback: ' . $e->getMessage());
    }
    return $out;
  }

  /** Normalize keywords: localized arrays → flat array of strings */
  public static function keywordsForPublication($publication, array $preferredLocales) {
    $kw = $publication->getData('keywords');

    if (is_array($kw)) {
      foreach ($preferredLocales as $loc) {
        if (!empty($kw[$loc]) && is_array($kw[$loc])) {
          return array_values(array_unique(array_filter(array_map('trim', $kw[$loc]))));
        }
      }
      // Fallback: merge all locale arrays
      $all = [];
      foreach ($kw as $arr) if (is_array($arr)) $all = array_merge($all, $arr);
      return array_values(array_unique(array_filter(array_map('trim', $all))));
    }

    if (is_string($kw) && $kw !== '') {
      $parts = preg_split('/[,;]+/', $kw);
      return array_values(array_unique(array_filter(array_map('trim', (array)$parts))));
    }

    return [];
  }

  /** Split "10.1234/abcd" → ["10.1234","abcd"] */
  public static function splitWdoi($wdoi) {
    $wdoi = (string)$wdoi;
    $pos = strpos($wdoi, '/');
    if ($pos === false) return [null, null];
    return [substr($wdoi, 0, $pos), substr($wdoi, $pos + 1)];
  }

  /** Build the Registry mint payload (no HTTP) */
  public static function buildMintPayload($plugin, $contextId, $publication, $submission, $request) {
    $locales = self::preferredLocales($request);

    $title       = self::pickLocalized($publication->getData('title'), $locales);
    $abstract    = self::pickLocalized($publication->getData('abstract'), $locales);

    // Resource type from plugin settings (optional)
    $resType = (string) ($plugin->getSetting($contextId, 'wdoiResType') ?? '');
    if ($resType === '') $resType = null;

    // Prefix & collection (from settings)
    $prefix       = (string) ($plugin->getSetting($contextId, 'wdoiPrefix') ?? '');
    $collectionId = (string) ($plugin->getSetting($contextId, 'wdoiCollection') ?? '');

    // Date issued: prefer publication date
    $issued = (string) ($publication->getData('datePublished') ?? '');
    if ($issued !== '') $issued = substr($issued, 0, 10); // YYYY-MM-DD

    // Authors & keywords
    $authors  = self::authorsForPublication((int)$publication->getId(), $contextId, $locales);
    $keywords = self::keywordsForPublication($publication, $locales);

    // Minimal, clean metadata JSON
    $metadataJson = array_filter([
      'title'    => $title ?: null,
      'abstract' => $abstract ?: null,
      'issued'   => $issued ?: null,
      'authors'  => $authors,
      'keywords' => $keywords,
    ], fn($v) => $v !== null && $v !== '' && $v !== []);

    // The plugin’s onPublish will add targetUrl.
    return array_filter([
      'prefix'       => $prefix ?: null,
      'suffixPolicy' => 'AUTO',
      'targetUrl'    => null,            // caller sets
      'resourceType' => $resType,        // may be null
      'collectionId' => $collectionId ?: null,
      'metadataJson' => (object)$metadataJson,
      '__reason'     => 'Mint via OJS publish',
    ], fn($v) => $v !== null);
  }

  /** Build metadata-only body for update PUT */
  public static function buildMetadataJson($publication, $submission, $request) {
    $locales = self::preferredLocales($request);

    $title    = self::pickLocalized($publication->getData('title'), $locales);
    $abstract = self::pickLocalized($publication->getData('abstract'), $locales);

    $issued = (string) ($publication->getData('datePublished') ?? '');
    if ($issued !== '') $issued = substr($issued, 0, 10);

    $ctxIdFromSubmission = $submission ? (int)$submission->getData('contextId') : null;
    $authors  = self::authorsForPublication((int)$publication->getId(), $ctxIdFromSubmission, $locales);
    $keywords = self::keywordsForPublication($publication, $locales);

    $metadata = array_filter([
      'title'    => $title ?: null,
      'abstract' => $abstract ?: null,
      'issued'   => $issued ?: null,
      'authors'  => $authors,
      'keywords' => $keywords,
    ], fn($v) => $v !== null && $v !== '' && $v !== []);

    return (object)$metadata;
  }
}
