<?php
class WdoiClient {
  private $base;
  private $key;
  private $connectTimeout;
  private $timeout;

  public function __construct($base, $key, $connectTimeoutSeconds = 5, $timeoutSeconds = 5) {
    $this->base = rtrim($base, '/');
    $this->key = $key;
    $this->connectTimeout = (int)$connectTimeoutSeconds;
    $this->timeout = (int)$timeoutSeconds;
  }

  /** Low-level request wrapper */
  private function doRequest($method, $path, $body = null) {
    $url = $this->base . $path;
    $ch = curl_init($url);

    $headers = [
      'Authorization: Bearer ' . $this->key,
      'Content-Type: application/json',
      'Accept: application/json',
      'User-Agent: wdoi-ojs-plugin/1.0',
    ];

    curl_setopt_array($ch, [
      CURLOPT_CUSTOMREQUEST   => $method,
      CURLOPT_RETURNTRANSFER  => true,
      CURLOPT_HTTPHEADER      => $headers,
      CURLOPT_CONNECTTIMEOUT  => $this->connectTimeout,
      CURLOPT_TIMEOUT         => $this->timeout,
      CURLOPT_FOLLOWLOCATION  => true,
    ]);

    if ($body !== null) {
      curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
    }

    $respBody = curl_exec($ch);
    $err      = curl_error($ch);
    $code     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // If curl itself failed
    if ($err) {
      return [
        'ok'     => false,
        'code'   => 0,
        'body'   => $respBody ?: '',
        'json'   => null,
        'error'  => $err,
        'result' => 'curl_error'
      ];
    }

    $json = null;
    $jsonError = null;
    if (is_string($respBody) && $respBody !== '') {
      $decoded = json_decode($respBody, true);
      if (json_last_error() === JSON_ERROR_NONE) {
        $json = $decoded;
      } else {
        $jsonError = json_last_error_msg();
      }
    }

    $ok = $code >= 200 && $code < 300;

    return [
      'ok'     => $ok,
      'code'   => $code,
      'body'   => $respBody ?: '',
      'json'   => $json,
      'error'  => $jsonError,
      'result' => $ok ? 'success' : ($jsonError ? 'json_error' : 'http_error')
    ];
  }

  /** Convenience wrappers */
  public function get($path)             { return $this->doRequest('GET',    $path); }
  public function post($path, $body)     { return $this->doRequest('POST',   $path, $body); }
  public function put($path, $body)      { return $this->doRequest('PUT',    $path, $body); }
  public function delete($path, $body=null) { return $this->doRequest('DELETE', $path, $body); }
}
